function [sub_img,bg_img,se_tmp] = bksub(varargin)
%This function allows you to calculate and remove the background of a
%grayscale image.  Essentually, it first erodes the image using a
%structuring element, then dilates using the same element to generate a
%background image.
%Synatax:   [ext_img,dil_img,se] = bksub('img',img,'se',20,'se_type','ball','threshold','30'); 
%Input:     img = the image or image stack that needs to be background
%                 subtracted.
%           'se' = is the parameters for the structuring element created by
%               the strel function: 
%                 SE = strel(shape, parameters)
%                 SE = strel('arbitrary', nhood)
%                 SE = strel('arbitrary', nhood, h)
%                 SE = strel('ball', r, h, n)
%                 SE = strel('diamond', r)
%                 SE = strel('disk', r, n)
%                 SE = strel('line', len, deg)
%                 SE = strel('octagon', r)
%                 SE = strel('pair', offset)
%                 SE = strel('periodicline', p, v)
%                 SE = strel('rectangle', mn)
%                 SE = strel('square', w)
%           'se_type' = is the structuring element type used.  Default =
%               ball.
%           'imgnum' = select the image you want to use to preview the
%               background subtraction.
%           'prev' = preview the subtraction.  Default = 0 (off) Note: only
%               works for ball type SE right now.
%           'sav' = save the new images or not, default = 1 (on)
%           'fullout' = use full output, meaning subtracted image plus
%               dilation image and structuring element.  Default = 1 (full)

[imgs,se,filenames,path,sav,prev,fullout] = parse(varargin);
%go through the images
for j = 1:size(imgs,2)
    img = imgs{1,j};    %pull one image set out
    if prev
        se_tmp = se{1,j};       %its corresponding structuring element
    else
        se_tmp = se{1,1};       %without preview there is only one se
    end
    %create the images
    sub_img = zeros(size(img),class(img));      %preallocate
    bg_img = zeros(size(img),class(img));
    for i = 1:size(img,3)       %go through the stack
        sub_img(:,:,i) = imtophat(img(:,:,i),se_tmp);
        if fullout==1
            bg_img(:,:,i) = imopen(img(:,:,i),se_tmp);
        end
    end
    
    %save the imags
    if sav
        warning 'off'
        if j==1     %make the roots once
            mkdir(path,'background_img');
            mkdir(path,'bksubtracted');      %create output directory
        end
        filename = filenames{1,j};  %current file
        mkdir([path,filesep,'bksubtracted'],filename(1:end-4));      %create output directory
        stk2tiff(sub_img,[filename(1:end-4),'_bksub'],[path,'bksubtracted',filesep,filename(1:end-4),filesep]);
        %now save out each image stack
        mkdir([path,filesep,'background_img'],filename(1:end-4));
        stk2tiff(bg_img,[filename(1:end-4),'_bkground'],[path,'background_img',filesep,filename(1:end-4),filesep]);
        warning 'on'
    end
end
    
       

%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [imgs,se,filenames,path,sav,prev,fullout] = parse(input)

se_type = 'ball';   %default ball structuring element.
imgs = [];       %set images empty for now
imgnum = [];        %the selected image number
prev = 0;       %set off
sav = 1;    %default on
filenames = [];
path = [];
fullout = 1;

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'prev'
                    prev = input{1,i+1};
                case 'sav'
                    sav = input{1,i+1};
                case 'imgnum'
                    imgnum = input{1,i+1};
                case 'se_type'
                    if ischar(input{1,i+1})
                        se_type = input{1,i+1};
                    else
                        warning(['Your entered structuring element type is not recognized, reverting to defalut type.']);
                    end
                case 'nhood'
                    nhood = input{1,i+1};
                case 'r'    %radius
                    r = input{1,i+1};
                case 'h'    %height
                    h = input{1,i+1};
                case 'n'
%                     When N is greater than 0, the ball-shaped structuring element is approximated by a sequence of N nonflat, 
%                     line-shaped structuring elements. When N equals 0, no approximation is used, and the structuring element 
%                     members consist of all pixels whose centers are no greater than R away from the origin. The corresponding 
%                     height values are determined from the formula of the ellipsoid specified by R and H. If N is not specified, 
%                     the default value is 8.
                    n = input{1,i+1};
                case 'len'    %length
                    len = input{1,i+1};
                case 'deg'
                    deg = input{1,i+1};
                case 'offset'
                    offset = input{1,i+1};
                case 'p'    %radius
                    p = input{1,i+1};
                case 'v'    %height
                    v = input{1,i+1};
                case 'mn'    %sides of the rectangle
                    mn = input{1,i+1};
                case 'w'    %sides of the square
                    w = input{1,i+1};
                case 'img'
                    imgs = input{1,i+1};
                case 'fullout'   %full output
                    fullout = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end
%create the structuring element
switch se_type
    case 'ball'     %note: if you want to use a parameter for the structuring element, you must enter all of the parameters.
        try
            se{1,1} = strel('ball',r,h,n);   %check to see if parameters are defined
        catch
            se{1,1} = strel('ball',10,10);     %if not default, n = 8 in this case.
        end
    case 'arbitrary'     
        try
            se{1,1} = strel('arbitrary',nhood,h);   %check to see if parameters are defined
        catch
            se{1,1} = strel('arbitrary',10,10);     %if not default.
        end
    case 'diamond'
        try
            se{1,1} = strel('diamond',r);   %check to see if parameters are defined
        catch
            se{1,1} = strel('diamond',10);     %if not default.
        end
    case 'disk'
        try
            se{1,1} = strel('disk',r,n);   %check to see if parameters are defined
        catch
            se{1,1} = strel('disk',10);     %if not. default n = 4
        end
    case 'line'
        try
            se{1,1} = strel('line',len,deg);   %check to see if parameters are defined
        catch
            se{1,1} = strel('line',10,0);     %if not default.
        end
    case 'octagon'
        try
            se{1,1} = strel('octagon',r);   %check to see if parameters are defined
        catch
            se{1,1} = strel('octagon',10);     %if not default.
        end
    case 'pair'
        try
            se{1,1} = strel('pair',offset);   %check to see if parameters are defined
        catch
            se{1,1} = strel('pair',[10 10]);     %if not default.
        end
    case 'periodicline'
        try
            se{1,1} = strel('periodicline',p,v);   %check to see if parameters are defined
        catch
            se{1,1} = strel('periodicline',10,[10 10]);     %if not default.
        end
    case 'rectangle'
        try
            se{1,1} = strel('rectangle',mn);   %check to see if parameters are defined
        catch
            se{1,1} = strel('rectangle',[10 10]);     %if not default. square!
        end
    case 'square'
        try
            se{1,1} = strel('square',w);
        catch
            se{1,1} = strel('square',10);
        end
end
%open the image stacks
if isempty(imgs)
    prompt_box('title','Open Original Image','prompt1','Select image stack ',...
        'prompt2','as the reference image','position','center');
    pause(0.25);
    [imgs,img_range,filenames,path] = stack_gui(1);  %use cell output mode.
end
%check if imgs is a cell, if not make it one
if ~iscell(imgs)
    imgs = {imgs};  %cell now
end
%OK if preview then display images
if prev
    for i = 1:size(imgs,2)
        img = imnorm(imgs{1,1});  %don't need cell for this, normalize for better performance.
        if size(img,3)==1   %there is only one image
            imgnum = [1 1];     %this is it
            hi_lo = 0;      %don't need to show high low, because well, 1 image buddy.
        else
            if isempty(imgnum)  %if the image number is not specified.
                %pick the image to use for display.  We want the one with the least
                %contrast
                for k = 1:size(img,3)   %step through the stack
                    img_std(k) = mean(std(single(img(:,:,k))));     %grab the standard deviation of the image
                    img_int(k) = sum(sum(single(img(:,:,k))));      %grab the total intensity of the image
                end
                img_std = img_std/mean(img_std);    %normalize
                img_int = img_int/mean(img_int);    %ditto
                img_score = img_std+img_int;    %create an image score
                imgnum = find(img_score==min(img_score));  %We want the lowest score
                imgnum(2) = find(img_score==max(img_score));  %and the highest score
            end
            hi_lo = 1;  %yes two please
        end
        se{1,i} = play_thresh(img(:,:,imgnum(1)),img(:,:,imgnum(2)),se{1,1},hi_lo);   %pick your threshold different se per image
    end
end
%--------------------------------------------------------------------------
function [se] = play_thresh(img,img2,se,hi_lo)
%this function lets the user select a threshold interactively
%if inital thresh is not given, default to 0.1
if nargin<3
    se = strel('ball',10,10);   %no dilation
    hi_lo = 1;
end
itr = 0;        %initiate iterate again or not switch
warning('off')
%show original
if hi_lo
    h3 = figure('Name','Original Img2','NumberTitle','off');
    imshow(img2);
    h2 = figure('Name','Subtracted Img2','NumberTitle','off');
    h5 = figure('Name','Background Img2','NumberTitle','off');   %background image
end
r = 10;     %default
h1 = figure('Name','Original Img','NumberTitle','off');
imshow(img);
h = figure('Name','Subtracted Img','NumberTitle','off');     %create image
h4 = figure('Name','Background Img','NumberTitle','off');   %for the background image
while itr==0
    img_bw1 = imtophat(img,se);
    img_bg1 = imopen(img,se);
    if hi_lo    %show second, because it is different
        img_bw2 = imtophat(img2,se);
        img_bg2 = imopen(img,se);
        figure(h2);     %select figure
        imshow(img_bw2);
        figure(h5);
        imshow(imnorm(img_bg2));
    end
    figure(h);     %select figure
    %hold
    imshow(img_bw1);
    figure(h4);
    imshow(imnorm(img_bg1));
    itr = yes_no_box('title','Is this image OK?','caption1','Select yes is you want to move on.',...
        'caption2','Otherwise enter the desired radius.','position','west');
    pause(0.25);
    if itr==0
        r = str2double(response_box('title','Enter your new ball radius','input',num2str(r),'position','center'));
        se = strel('ball',r,r);
        pause(0.25);
    end
end
close(h1)
close(h)
close(h4)
if hi_lo
    close(h2)
    close(h3)
    close(h5)
end
warning('on')